<?php

namespace App\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Crypt;
use Exception;

class LicenseDatabaseService
{
    protected $dbPath;
    protected $connectionName = 'license';
    protected $dbPassword;

    public function __construct()
    {
        // مسار قاعدة بيانات SQLite للتراخيص
        $this->dbPath = storage_path('app/licenses.db');
        
        // كلمة مرور قاعدة البيانات (من .env أو قيمة افتراضية)
        $this->dbPassword = env('LICENSE_DB_PASSWORD', 'CHANGE_THIS_SECRET_PASSWORD');
        
        // إنشاء الاتصال إذا لم يكن موجوداً
        $this->setupConnection();
    }

    /**
     * إعداد اتصال قاعدة بيانات SQLite منفصلة
     */
    protected function setupConnection(): void
    {
        // إنشاء مجلد إذا لم يكن موجوداً
        $dir = dirname($this->dbPath);
        if (!File::exists($dir)) {
            File::makeDirectory($dir, 0755, true);
        }

        // إنشاء ملف قاعدة البيانات إذا لم يكن موجوداً
        if (!File::exists($this->dbPath)) {
            File::put($this->dbPath, '');
        }

        // إضافة الاتصال إلى config
        config([
            "database.connections.{$this->connectionName}" => [
                'driver' => 'sqlite',
                'database' => $this->dbPath,
                'prefix' => '',
                'foreign_key_constraints' => true,
            ],
        ]);
        
        // حماية الملف بصلاحيات النظام (على Linux/Mac)
        if (file_exists($this->dbPath) && PHP_OS_FAMILY !== 'Windows') {
            @chmod($this->dbPath, 0600);
        }

        // إنشاء الجدول إذا لم يكن موجوداً
        $this->createTableIfNotExists();
    }

    /**
     * إنشاء جدول التراخيص في قاعدة البيانات المنفصلة
     */
    protected function createTableIfNotExists(): void
    {
        try {
            DB::connection($this->connectionName)->statement("
                CREATE TABLE IF NOT EXISTS licenses (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    license_key TEXT NOT NULL,
                    license_key_hash TEXT NOT NULL,
                    serial_number TEXT NOT NULL,
                    serial_number_hash TEXT NOT NULL,
                    domain TEXT,
                    machine_id TEXT,
                    machine_name TEXT,
                    ip_address TEXT,
                    email TEXT,
                    customer_name TEXT,
                    status TEXT DEFAULT 'pending',
                    activated_at TEXT,
                    expires_at TEXT,
                    max_users INTEGER DEFAULT 10,
                    notes TEXT,
                    created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                    updated_at TEXT DEFAULT CURRENT_TIMESTAMP
                )
            ");

            // إنشاء فهارس
            DB::connection($this->connectionName)->statement("
                CREATE INDEX IF NOT EXISTS idx_license_key_hash ON licenses(license_key_hash)
            ");
            DB::connection($this->connectionName)->statement("
                CREATE INDEX IF NOT EXISTS idx_serial_number_hash ON licenses(serial_number_hash)
            ");
            DB::connection($this->connectionName)->statement("
                CREATE INDEX IF NOT EXISTS idx_machine_id ON licenses(machine_id)
            ");
            DB::connection($this->connectionName)->statement("
                CREATE INDEX IF NOT EXISTS idx_status ON licenses(status)
            ");
        } catch (Exception $e) {
            \Log::error('Error creating license database: ' . $e->getMessage());
        }
    }

    public function getConnection()
    {
        return DB::connection($this->connectionName);
    }

    /**
     * إدراج ترخيص جديد
     */
    public function insert(array $data): bool
    {
        try {
            $licenseKeyHash = hash('sha256', strtoupper($data['license_key'] ?? ''));
            $serialNumberHash = hash('sha256', strtoupper($data['serial_number'] ?? ''));
            
            $encryptedData = $this->encryptSensitiveData($data);
            $encryptedData['license_key_hash'] = $licenseKeyHash;
            $encryptedData['serial_number_hash'] = $serialNumberHash;
            $encryptedData['created_at'] = now()->toDateTimeString();
            $encryptedData['updated_at'] = now()->toDateTimeString();
            
            return $this->getConnection()->table('licenses')->insert($encryptedData);
        } catch (Exception $e) {
            \Log::error('Error inserting license: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * البحث عن ترخيص
     */
    public function find(string $licenseKey, ?string $serialNumber = null): ?array
    {
        try {
            $licenseKeyHash = hash('sha256', strtoupper($licenseKey));
            
            $query = $this->getConnection()->table('licenses')
                ->where('license_key_hash', $licenseKeyHash);
            
            if ($serialNumber) {
                $serialNumberHash = hash('sha256', strtoupper($serialNumber));
                $query->where('serial_number_hash', $serialNumberHash);
            }
            
            $license = $query->first();
            
            if (!$license) {
                return null;
            }
            
            return $this->decryptSensitiveData((array)$license);
        } catch (Exception $e) {
            \Log::error('Error finding license: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * تحديث ترخيص
     */
    public function update(string $licenseKey, array $data): bool
    {
        try {
            $licenseKeyHash = hash('sha256', strtoupper($licenseKey));
            
            $license = $this->getConnection()->table('licenses')
                ->where('license_key_hash', $licenseKeyHash)
                ->first();
            
            if (!$license) {
                return false;
            }
            
            $currentData = $this->decryptSensitiveData((array)$license);
            $mergedData = array_merge($currentData, $data);
            $encryptedData = $this->encryptSensitiveData($mergedData);
            
            if (isset($data['license_key'])) {
                $encryptedData['license_key_hash'] = hash('sha256', strtoupper($data['license_key']));
            }
            if (isset($data['serial_number'])) {
                $encryptedData['serial_number_hash'] = hash('sha256', strtoupper($data['serial_number']));
            }
            
            $encryptedData['updated_at'] = now()->toDateTimeString();
            
            $this->getConnection()->table('licenses')
                ->where('id', $license->id)
                ->update($encryptedData);
            
            return true;
        } catch (Exception $e) {
            \Log::error('Error updating license: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * البحث عن ترخيص حسب Machine ID
     */
    public function findByMachineId(string $machineId): ?array
    {
        try {
            // التأكد من أن الجدول موجود
            $this->createTableIfNotExists();
            
            $license = $this->getConnection()->table('licenses')
                ->where('machine_id', $machineId)
                ->where('status', 'active')
                ->first();
            
            if (!$license) {
                return null;
            }
            
            return $this->decryptSensitiveData((array)$license);
        } catch (Exception $e) {
            \Log::error('Error finding license by machine ID: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * تشفير البيانات الحساسة
     */
    protected function encryptSensitiveData(array $data): array
    {
        $sensitiveFields = ['license_key', 'serial_number', 'email', 'customer_name', 'notes'];
        
        foreach ($sensitiveFields as $field) {
            if (isset($data[$field]) && $data[$field] !== null) {
                $data[$field] = Crypt::encryptString($data[$field]);
            }
        }
        
        return $data;
    }

    /**
     * فك تشفير البيانات الحساسة
     */
    protected function decryptSensitiveData(array $data): array
    {
        $sensitiveFields = ['license_key', 'serial_number', 'email', 'customer_name', 'notes'];
        
        foreach ($sensitiveFields as $field) {
            if (isset($data[$field]) && $data[$field] !== null) {
                try {
                    $data[$field] = Crypt::decryptString($data[$field]);
                } catch (Exception $e) {
                    // إذا فشل فك التشفير، قد يكون غير مشفر (للتوافق مع البيانات القديمة)
                }
            }
        }
        
        return $data;
    }
}



