<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Services\LicenseService;
use App\Services\TrialService;
use Symfony\Component\HttpFoundation\Response;

class CheckLicense
{
    protected $licenseService;
    protected $trialService;

    public function __construct(LicenseService $licenseService, TrialService $trialService)
    {
        $this->licenseService = $licenseService;
        $this->trialService = $trialService;
    }

    public function handle(Request $request, Closure $next): Response
    {
        $routeName = $request->route()?->getName();
        $path = $request->path();

        // السماح بصفحات الترخيص والفترة التجريبية
        if ($routeName && (str_starts_with($routeName, 'license.') || str_starts_with($routeName, 'trial.'))) {
            return $next($request);
        }

        // السماح بصفحات المصادقة
        if ($routeName && (in_array($routeName, ['login', 'logout']) || str_starts_with($routeName, 'password.'))) {
            return $next($request);
        }

        // السماح بصفحة اللغة
        if ($routeName === 'locale.switch') {
            return $next($request);
        }

        // السماح بصفحة الرئيسية
        if ($path === '/' || $path === '') {
            return $next($request);
        }

        // التحقق من الترخيص أولاً
        try {
            $hasLicense = $this->licenseService->verifyLocal();
            if ($hasLicense) {
                return $next($request);
            }
        } catch (\Exception $e) {
            \Log::error('License check error: ' . $e->getMessage());
            // في حالة الخطأ، نتابع للتحقق من الفترة التجريبية
        }

        // التحقق من الفترة التجريبية
        try {
            $isExpired = $this->trialService->isExpired();
            $trialInfo = $this->trialService->getTrialInfo();
        } catch (\Exception $e) {
            \Log::error('Trial check error: ' . $e->getMessage());
            // في حالة الخطأ، نسمح بالوصول لتجنب حجب النظام
            return $next($request);
        }
        
        if (!$isExpired) {
            // عرض تحذير قبل 3 أيام من الانتهاء
            if ($this->trialService->isExpiringSoon()) {
                $daysLeft = $this->trialService->getDaysRemaining();
                if ($request->session()->get('trial_warning_shown') !== true) {
                    $request->session()->flash('trial_warning', [
                        'message' => "تنبيه: الفترة التجريبية ستنتهي خلال {$daysLeft} يوم/أيام. يرجى تفعيل الترخيص للاستمرار.",
                        'days_left' => $daysLeft
                    ]);
                    $request->session()->put('trial_warning_shown', true);
                }
            }
            
            return $next($request);
        }

        // إذا انتهت الفترة التجريبية وليس هناك ترخيص
        if ($request->expectsJson()) {
            return response()->json([
                'message' => 'انتهت الفترة التجريبية. يرجى تفعيل الترخيص.',
                'expired' => true,
                'license_required' => true
            ], 403);
        }

        return redirect()->route('trial.expired');
    }
}



