<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Cashbox;
use App\Models\FinancialTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TransactionController extends Controller
{
    public function index(Request $request)
    {
        $transactions = FinancialTransaction::query()
            ->with('cashbox')
            ->when($request->filled('type'), fn ($query) => $query->where('type', $request->string('type')))
            ->when($request->filled('from'), fn ($query) => $query->whereDate('occurred_on', '>=', $request->date('from')))
            ->when($request->filled('to'), fn ($query) => $query->whereDate('occurred_on', '<=', $request->date('to')))
            ->orderByDesc('occurred_on')
            ->paginate(20)
            ->withQueryString();

        return view('admin.transactions.index', compact('transactions'));
    }

    public function create()
    {
        $cashboxes = Cashbox::all();

        return view('admin.transactions.create', compact('cashboxes'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'type' => ['required', 'in:income,expense'],
            'category' => ['required', 'string', 'max:120'],
            'amount' => ['required', 'numeric', 'min:0.01'],
            'description' => ['nullable', 'string'],
            'occurred_on' => ['required', 'date'],
            'cashbox_id' => ['required', 'exists:cashboxes,id'],
        ]);

        DB::transaction(function () use ($data) {
            $transaction = FinancialTransaction::create([
                'type' => $data['type'],
                'category' => $data['category'],
                'reference_type' => null,
                'reference_id' => null,
                'cashbox_id' => $data['cashbox_id'],
                'amount' => $data['amount'],
                'description' => $data['description'] ?? null,
                'occurred_on' => $data['occurred_on'],
            ]);

            $cashbox = Cashbox::find($data['cashbox_id']);

            if ($cashbox) {
                $direction = $data['type'] === 'income' ? 'in' : 'out';
                $cashbox->recordEntry($data['amount'], $direction, $transaction);
            }
        });

        return redirect()->route('admin.transactions.index')->with('status', __('app.transactions.created'));
    }

    public function show(FinancialTransaction $transaction)
    {
        return view('admin.transactions.show', compact('transaction'));
    }

    public function edit(FinancialTransaction $transaction)
    {
        if ($transaction->reference_type) {
            return redirect()->route('admin.transactions.show', $transaction)
                ->withErrors(__('app.transactions.locked_error'));
        }

        return view('admin.transactions.edit', compact('transaction'));
    }

    public function update(Request $request, FinancialTransaction $transaction)
    {
        if ($transaction->reference_type) {
            return redirect()->route('admin.transactions.show', $transaction)
                ->withErrors(__('app.transactions.locked_error'));
        }

        $data = $request->validate([
            'category' => ['required', 'string', 'max:120'],
            'description' => ['nullable', 'string'],
        ]);

        $transaction->update($data);

        return redirect()->route('admin.transactions.show', $transaction)->with('status', __('app.transactions.updated'));
    }

    public function destroy(FinancialTransaction $transaction)
    {
        if ($transaction->reference_type) {
            return redirect()->back()->withErrors(__('app.transactions.cannot_delete'));
        }

        if (! $transaction->reference_type && $transaction->cashbox) {
            $direction = $transaction->type === 'income' ? 'out' : 'in';
            $transaction->cashbox->recordEntry($transaction->amount, $direction);
        }

        $transaction->delete();

        return redirect()->route('admin.transactions.index')->with('status', __('app.transactions.deleted'));
    }
}
