<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Cashbox;
use App\Models\FinancialTransaction;
use App\Models\ServiceOrder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ServiceOrderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $orders = ServiceOrder::query()
            ->with('items.clothingItem')
            ->when($request->filled('status'), fn ($query) => $query->where('status', $request->string('status')))
            ->when($request->filled('search'), function ($query) use ($request) {
                $term = '%' . $request->string('search') . '%';
                $query->where(function ($subQuery) use ($term) {
                    $subQuery->where('order_number', 'like', $term)
                        ->orWhere('barcode', 'like', $term)
                        ->orWhere('customer_name', 'like', $term);
                });
            })
            ->orderByDesc('created_at')
            ->paginate(15)
            ->withQueryString();

        return view('admin.orders.index', compact('orders'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return redirect()->route('service-orders.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        return redirect()->route('service-orders.index');
    }

    /**
     * Display the specified resource.
     */
    public function show(ServiceOrder $order)
    {
        $order->load('items.clothingItem');

        return view('admin.orders.show', ['serviceOrder' => $order]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ServiceOrder $order)
    {
        return redirect()->route('admin.orders.show', $order);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ServiceOrder $order)
    {
        $data = $request->validate([
            'status' => ['nullable', 'in:processing,ready,collected'],
            'payment_status' => ['nullable', 'in:unpaid,paid'],
            'mark_paid' => ['sometimes', 'boolean'],
        ]);

        DB::transaction(function () use ($order, $data, $request) {
            $updates = [];

            if (isset($data['status'])) {
                $updates['status'] = $data['status'];

                if ($data['status'] === 'collected') {
                    $updates['collected_at'] = now();
                }
            }

            if (($data['payment_status'] ?? null) === 'paid' || $request->boolean('mark_paid')) {
                if ($order->payment_status !== 'paid') {
                    $this->registerPayment($order);
                    $updates['payment_status'] = 'paid';
                }
            }

            if (! empty($updates)) {
                $order->update($updates);
            }
        });

        if ($request->expectsJson() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => __('app.orders.show.updated'),
                'order' => $order->fresh(),
            ]);
        }

        return redirect()->back()->with('status', __('app.orders.show.updated'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ServiceOrder $order)
    {
        // Check permission
        if (!auth()->user()->hasPermission('manage_orders')) {
            abort(403, 'Unauthorized action.');
        }

        $orderNumber = $order->order_number;
        $order->delete();

        return redirect()->route('admin.orders.index')->with('status', __('app.orders.show.deleted', ['number' => $orderNumber]));
    }

    protected function registerPayment(ServiceOrder $order): void
    {
        if ($order->total <= 0) {
            return;
        }

        $cashbox = Cashbox::first();

        if (! $cashbox) {
            return;
        }

        $transaction = FinancialTransaction::create([
            'type' => 'income',
            'category' => 'laundry-service',
            'reference_type' => ServiceOrder::class,
            'reference_id' => $order->id,
            'cashbox_id' => $cashbox->id,
            'amount' => $order->total,
            'description' => 'Payment recorded via admin panel',
            'occurred_on' => now(),
        ]);

        $cashbox->recordEntry($order->total, 'in', $transaction);
    }
}
