<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ClothingItem;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ClothingItemController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $items = ClothingItem::orderBy('name')->paginate(12);

        return view('admin.items.index', compact('items'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.items.create', ['clothingItem' => new ClothingItem()]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $data = $this->validateItem($request);

        $data['slug'] = $data['slug'] ?? Str::slug($data['name']);

        if ($request->hasFile('image')) {
            $data['image_path'] = $this->uploadImage($request->file('image'), $data['slug']);
        }

        ClothingItem::create($data);

        return redirect()->route('admin.items.index')->with('status', __('app.items.created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(ClothingItem $item)
    {
        return redirect()->route('admin.items.edit', $item);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ClothingItem $item)
    {
        return view('admin.items.edit', ['clothingItem' => $item]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ClothingItem $item)
    {
        $data = $this->validateItem($request, $item->id);

        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($item->image_path && file_exists(public_path('images/' . $item->image_path))) {
                @unlink(public_path('images/' . $item->image_path));
            }
            $data['image_path'] = $this->uploadImage($request->file('image'), $data['slug'] ?? $item->slug);
        }

        $item->update($data);

        return redirect()->route('admin.items.index')->with('status', __('app.items.updated'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ClothingItem $item)
    {
        // Delete image if exists
        if ($item->image_path && file_exists(public_path('images/' . $item->image_path))) {
            @unlink(public_path('images/' . $item->image_path));
        }

        $item->delete();

        return redirect()->route('admin.items.index')->with('status', __('app.items.deleted'));
    }

    protected function validateItem(Request $request, ?int $itemId = null): array
    {
        $rules = [
            'name' => ['required', 'string', 'max:120'],
            'slug' => ['nullable', 'string', 'max:120', 'unique:clothing_items,slug,' . $itemId],
            'default_wash_price' => ['required', 'numeric', 'min:0'],
            'default_iron_price' => ['required', 'numeric', 'min:0'],
            'is_active' => ['sometimes', 'boolean'],
        ];

        if ($itemId) {
            $rules['image'] = ['nullable', 'image', 'mimes:jpeg,jpg,png,gif,webp', 'max:2048'];
        } else {
            $rules['image'] = ['required', 'image', 'mimes:jpeg,jpg,png,gif,webp', 'max:2048'];
        }

        $data = $request->validate($rules);

        $data['is_active'] = $request->boolean('is_active', true);

        return $data;
    }

    protected function uploadImage($file, $slug): string
    {
        $directory = public_path('images');
        if (!file_exists($directory)) {
            mkdir($directory, 0755, true);
        }

        $extension = $file->getClientOriginalExtension();
        $filename = $slug . '_' . time() . '.' . $extension;
        $file->move($directory, $filename);

        return $filename;
    }
}
