<?php

namespace App\Console\Commands;

use App\Services\LicenseDatabaseService;
use Illuminate\Console\Command;
use Illuminate\Support\Str;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xls;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;

class GenerateLicense extends Command
{
    protected $signature = 'license:generate {count=1 : Number of licenses to generate} {--excel : Export to Excel file}';
    protected $description = 'Generate license keys and optionally export to Excel';

    public function handle(LicenseDatabaseService $licenseDb)
    {
        $count = (int) $this->argument('count');
        $exportExcel = $this->option('excel');

        $this->info("Generating {$count} license(s)...");

        $licenses = [];
        $bar = $this->output->createProgressBar($count);
        $bar->start();

        for ($i = 0; $i < $count; $i++) {
            $licenseKey = $this->generateLicenseKey();
            $serialNumber = $this->generateSerialNumber();

            $licenseDb->insert([
                'license_key' => $licenseKey,
                'serial_number' => $serialNumber,
                'status' => 'pending',
                'max_users' => 10,
            ]);

            $licenses[] = [
                'no' => $i + 1,
                'license_key' => $licenseKey,
                'serial_number' => $serialNumber,
                'status' => 'Pending',
            ];

            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        // Export to Excel if requested
        if ($exportExcel) {
            $this->info('Exporting to Excel file...');
            $excelPath = $this->exportToExcel($licenses);
            $this->info("✓ Excel file created: <fg=green>{$excelPath}</>");
            $this->newLine();
        }

        $this->info("✓ Successfully generated {$count} license(s)");

        // Show first 10 licenses in console
        if ($count <= 10) {
            $this->newLine();
            foreach ($licenses as $license) {
                $this->line("  {$license['no']}. License Key: <fg=cyan>{$license['license_key']}</>");
                $this->line("     Serial Number: <fg=cyan>{$license['serial_number']}</>");
            }
        } else {
            $this->newLine();
            $this->line("  Showing first 10 licenses (see Excel file for all):");
            foreach (array_slice($licenses, 0, 10) as $license) {
                $this->line("  {$license['no']}. License Key: <fg=cyan>{$license['license_key']}</>");
                $this->line("     Serial Number: <fg=cyan>{$license['serial_number']}</>");
            }
        }

        return Command::SUCCESS;
    }

    protected function generateLicenseKey(): string
    {
        return strtoupper(
            Str::random(4) . '-' . 
            Str::random(4) . '-' . 
            Str::random(4) . '-' . 
            Str::random(4)
        );
    }

    protected function generateSerialNumber(): string
    {
        return 'SN-' . strtoupper(Str::random(12));
    }

    protected function exportToExcel(array $licenses): string
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Licenses');

        // Headers
        $headers = ['#', 'License Key', 'Serial Number', 'Status'];
        $sheet->fromArray($headers, null, 'A1');

        // Style header row
        $headerStyle = [
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF'],
                'size' => 12,
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4472C4'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => ['rgb' => '000000'],
                ],
            ],
        ];
        $sheet->getStyle('A1:D1')->applyFromArray($headerStyle);
        $sheet->getRowDimension(1)->setRowHeight(25);

        // Data
        $row = 2;
        foreach ($licenses as $license) {
            $sheet->setCellValue('A' . $row, $license['no']);
            $sheet->setCellValue('B' . $row, $license['license_key']);
            $sheet->setCellValue('C' . $row, $license['serial_number']);
            $sheet->setCellValue('D' . $row, $license['status']);

            // Style data rows
            $dataStyle = [
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => ['rgb' => 'CCCCCC'],
                    ],
                ],
            ];
            $sheet->getStyle('A' . $row . ':D' . $row)->applyFromArray($dataStyle);

            // Alternate row colors
            if ($row % 2 == 0) {
                $sheet->getStyle('A' . $row . ':D' . $row)->getFill()
                    ->setFillType(Fill::FILL_SOLID)
                    ->getStartColor()->setRGB('F2F2F2');
            }

            $row++;
        }

        // Auto-size columns
        foreach (range('A', 'D') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        // Set column widths
        $sheet->getColumnDimension('A')->setWidth(8);
        $sheet->getColumnDimension('B')->setWidth(25);
        $sheet->getColumnDimension('C')->setWidth(20);
        $sheet->getColumnDimension('D')->setWidth(15);

        // Freeze header row
        $sheet->freezePane('A2');

        // Save file
        $filename = 'licenses_' . date('Y-m-d_His') . '.xls';
        $filePath = storage_path('app/' . $filename);
        $writer = new Xls($spreadsheet);
        $writer->save($filePath);

        return $filePath;
    }
}
