<?php

namespace App\Console\Commands;

use App\Models\ActivationKey;
use Illuminate\Console\Command;
use Illuminate\Support\Str;

class GenerateActivationKeys extends Command
{
    protected $signature = 'activation:generate {count=500 : Number of keys to generate} {days=30 : Number of days for trial period}';
    protected $description = 'Generate activation keys and export to CSV/Excel';

    public function handle()
    {
        $count = (int) $this->argument('count');
        $days = (int) $this->argument('days');

        $this->info("Generating {$count} activation keys with {$days} days trial period...");

        $keys = [];
        $bar = $this->output->createProgressBar($count);
        $bar->start();

        for ($i = 0; $i < $count; $i++) {
            $key = $this->generateKey();
            
            ActivationKey::create([
                'key' => $key,
                'days' => $days,
                'is_used' => false,
            ]);

            $keys[] = [
                'key' => $key,
                'days' => $days,
                'status' => 'Unused',
            ];

            $bar->advance();
        }

        $bar->finish();
        $this->newLine();

        // Generate CSV file (can be opened in Excel)
        $this->info('Generating CSV file...');
        $this->generateCSV($keys, $days);

        $this->info("✓ Successfully generated {$count} activation keys!");
        $this->info("✓ CSV file created: storage/app/activation_keys.csv (can be opened in Excel)");

        return Command::SUCCESS;
    }

    protected function generateKey(): string
    {
        do {
            $key = strtoupper(Str::random(4) . '-' . Str::random(4) . '-' . Str::random(4) . '-' . Str::random(4));
        } while (ActivationKey::where('key', $key)->exists());

        return $key;
    }

    protected function generateCSV(array $keys, int $days): void
    {
        $filePath = storage_path('app/activation_keys.csv');
        $file = fopen($filePath, 'w');

        // Add BOM for Excel UTF-8 support
        fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));

        // Write headers
        fputcsv($file, ['Activation Key', 'Days', 'Status']);

        // Write data
        foreach ($keys as $keyData) {
            fputcsv($file, [
                $keyData['key'],
                $keyData['days'],
                $keyData['status'],
            ]);
        }

        fclose($file);
    }
}
